#include <opencv/cv.h>
#include <opencv/highgui.h>
#include <string>
#include <iostream>
#include <vector>
#include <sstream>
#include <algorithm>
#include <numeric>
#include <math.h>
#include <stdio.h>
int radius = 8;
int k = 64;
int c = 3;
float qtable[] = { 16, 11, 12, 14, 12, 10, 16, 14, 13, 14, 18, 17, 16, 19, 24,
		40, 26, 24, 22, 22, 24, 49, 35, 37, 29, 40, 58, 51, 61, 60, 57, 51, 56,
		55, 64, 72, 92, 78, 64, 68, 87, 69, 55, 56, 80, 109, 81, 87, 95, 98,
		103, 104, 103, 62, 77, 113, 121, 112, 100, 120, 92, 101, 103, 99 };

cv::Mat getPatch(cv::Mat& image, int row, int col) {
	int width = image.cols;
	int height = image.rows;
	int rowCount = height / radius;
	int colCount = width / radius;
	if (row >= rowCount || col > colCount) {
		return cv::Mat();
	} else {
		return cv::Mat(image, cv::Rect(radius * col, radius * row, radius,
				radius));
	}
}

double computeSaliency(cv::Mat& image, cv::Mat& imageGray, int row, int col) {
	int width = image.cols;
	int height = image.rows;
	cv::Mat currentPatch = getPatch(image, row, col);
	cv::Mat currentGrayPatch = getPatch(imageGray, row, col);
	cv::Mat fCurrentPatch;
	cv::dct(currentGrayPatch, fCurrentPatch);
	cv::Mat qTable(8, 8, CV_32FC1, qtable);
	cv::divide(fCurrentPatch, qTable, fCurrentPatch);
	std::vector<double> dArray;
	for (int i = 0; i < height / radius; i++) {
		if (i == row) {
			continue;
		}
		for (int j = 0; j < width / radius; j++) {
			if (j == col) {
				continue;
			}
			cv::Mat patch = getPatch(image, i, j);
			cv::Mat grayPatch = getPatch(imageGray, i, j);
			cv::Mat fPatch;
			cv::dct(grayPatch, fPatch);
			cv::divide(fPatch, qTable, fPatch);
			cv::Mat diff;
			cv::absdiff(currentPatch, patch, diff);
			cv::Mat fDiff;
			cv::absdiff(fCurrentPatch, fPatch, fDiff);
			cv::Scalar sum = cv::sum(diff);
			cv::Scalar fSum = cv::sum(fDiff);
			double dColor = sum.val[0] + sum.val[1] + sum.val[2];
			double dFreq = fSum.val[0];
			//std::cout << dFreq << std::endl;
			dFreq /= 5.0;
			dColor /= 255.0 * radius * radius;
			double dPosition = (row - i) * (row - i) + (col - j) * (col - j);
			dPosition = sqrt(dPosition) * radius;
			dPosition /= std::max(width, height);
			double d = dColor*dFreq  / (1 + c * dPosition);
			dArray.push_back(d);
		}
	}
	sort(dArray.begin(), dArray.end());
	double sum;
	if (dArray.size() < k) {
		sum = std::accumulate(dArray.begin(), dArray.end(), 0.0);
	} else {
		std::vector<double>::iterator iter = dArray.begin();
		sum = std::accumulate(iter, iter + k, 0.0);
	}
	double s = 1 - exp(-1.0 / k * sum);
	return s;
}

int main(int argc, char* argv[]) {
	std::string imagePath;
	if (argc == 1) {
		imagePath = "/home/zxwang/Desktop/dive.jpg";
	} else {
		imagePath = std::string(argv[1]);
	}
	cv::Mat image = cv::imread(imagePath, 1);
	cv::Mat imageLab;
	cv::Mat imageGray;
	cv::cvtColor(image, imageLab, CV_BGR2Lab);
	cv::cvtColor(image, imageGray, CV_BGR2GRAY);
	cv::Mat imageGrayFloat;
	imageGray.convertTo(imageGrayFloat, CV_32FC1, 1.0, 0.0);
	int width = imageLab.cols;
	int height = imageLab.rows;
	cv::Mat saliency = cv::Mat::zeros(height, width, CV_8UC1);
	for (int i = 0; i < height / radius; i++) {
		for (int j = 0; j < width / radius; j++) {
			double s = computeSaliency(imageLab, imageGrayFloat, i, j);
			cv::Mat patch(saliency, cv::Rect(j * radius, i * radius, radius,
					radius));
			patch.setTo(cv::Scalar((unsigned char) (s * 255)));
		}
	}
	double maxVal, minVal;
	cv::minMaxLoc(saliency, &minVal, &maxVal);
	saliency = saliency * (255.0 / maxVal);
	cv::GaussianBlur(saliency, saliency, cv::Size(9, 9), 5, 5);
	cv::namedWindow("test");
	cv::imshow("test", saliency);
	cv::waitKey(0);
	cv::imwrite("/home/zxwang/Desktop/saliency.png", saliency);
	return 0;
}
