function isa = sensorSet(isa,param,val,varargin)
%Set ISET sensor parameters.
%
%   isa = sensorSet(isa,param,val,varargin);
%
% This routine sets the ISET sensor parameters.  See sensorGet for the
% longer list of derived parameters.
%
% Examples:
%    isa = sensorSet(isa,'ColorOrder',colorOrder);
%    isa = sensorSet(isa,'PIXEL',pixel);
%    isa = sensorSet(isa,'autoExposure',1);   ('on' and 'off' work, too)
%    isa = sensorSet(isa,'sensorComputeMethod',baseName);
%    isa = sensorSet(isa,'quantizationMethod','10 bit');
%    isa = sensorSet(isa,'analogGain',5);
%
% General
%      {'name'} - This sensor's identifier
%      {'rows'} - number of rows
%      {'cols'} - number of cols
%      {'size'} - [rows,cols]
%
% Color
%      {'color'}  - structure containing color information
%        {'filterspectra'}    - color filter transmissivities
%        {'filternames'}      - color filter names
%        {'infraredfilter'}   - IR filter transmissivity
%      {'spectrum'}           - wavelength spectrum structure
%        {'wavelength'}       - wavelength samples
%      {'colorfilterarray'}   - color filter array structure
%        {'cfapattern'}       - color filter pattern
%        {'cfaunitblock'}     - basic color block
%
% Electrical and imaging
%      {'data'}               - data structure
%        {'volts'}            - voltage responses
%        {'digitalValues'}    - digital values
%      {'analogGain'}         - Transform volts
%      {'analogOffset'}       - Transform volts
%            Formula for offset and gain: (v + analogOffset)/analogGain)
%
%      {'roi'}                - region of interest information
%      {'exposureTime'}       - exposure time in seconds
%      {'autoExposure'}       - auto-exposure flag, 1 or 0, 'on' and 'off' OK
%      {'cds'}                - correlated double sampling flat
%      {'quantizationmethod'} - method used for quantization
%      {'dsnuimage'}          - Dark signal non uniformity (DSNU) image
%      {'prnuimage'}          - Photo response non uniformity (PRNU) image
%      {'dsnulevel'}          - dark signal nonuniformity (std dev)
%      {'prnulevel'}          - photoresponse nonuniformity (std dev)
%      {'columnfpnparameters'}- column fpn parameters, both offset and gain
%      {'colgainfpnvector'}   - column gain fpn data
%      {'coloffsetfpnvector'} - column offset fpn data
%      {'shotNoiseFlag'}      - include shot noise or not (default is yes)
%
% Pixel
%      {'pixel'}
%
% Optics
%      {'pixelvignetting'}
%      {'microlens'}
%      {'sensoretendue'}
%      {'microlensoffset'}  - used with microlens window toolbox
%
% Computational method
%      {'sensorcomputemethod'}- algorithm used to compute
%      {'ngridsamples'}       - number of spatial grid samples within a pixel
%
% Check for consistency between GUI and data
%      {'consistency'}
%
% Miscellaneous
%     {'mccRectHandles'}  - Handles for the rectangle selections in an MCC
%
% Private
%      {'editfilternames'}
%
% Copyright ImagEval Consultants, LLC, 2003.

if ieNotDefined('param'), error('Parameter field required.'); end

% Empty is an allowed value.  So we don't use ieNotDefined.
if ~exist('val','var'),   error('Value field required.'); end

switch lower(param)

    case {'name','title'}
        isa.name = val;

    case {'rows','row'}
        % We  always clear the data at this point to prevent any
        % possibility of a mis-match.
        isa.rows = val;
        isa = sensorClearData(isa);
    case {'cols','col'}
        % We  always clear the data at this point to prevent any
        % possibility of a mis-match.
        isa.cols = val;
        isa = sensorClearData(isa);
    case {'size'}
        % isa = sensorSet(isa,'size',[r c]);
        isa = sensorSet(isa,'rows',val(1));
        isa = sensorSet(isa,'cols',val(2));
        isa = sensorClearData(isa);

    case 'color'
        isa.color = val;
    case {'filterspectra','colorfilters'}
        isa.color.filterSpectra = val;
    case {'filternames','filtername'}
        isa.color.filterNames = val;
    case {'editfilternames','editfiltername'}
        % isa = sensorSet(isa,'editFilterNames',filterIndex,newFilterName);
        % This call either edits the name of an existing filter (if
        % whichFilter < length(filterNames), or adds a new filter
        %
        isa = sensorSet(isa,'filterNames',ieSetFilterName(val,varargin{1},isa));
    case {'infrared','irfilter','infraredfilter'}
        isa.color.irFilter = val(:);

    case {'spectrum'}
        isa.spectrum = val;
    case {'wavelength','wave','wavelengthsamples'}
        isa.spectrum.wave = val(:);
        %     case {'binwidth'}
        %         isa.spectrum.binwidth = val;

    case {'integrationtime','exptime','exposuretime','expduration','exposureduration'}
        % Seconds
        isa.integrationTime = val;
        isa.AE = 0;
    case {'autoexp','autoexposure','automaticexposure'}
        if ischar(val)
            % Let the person put on and off into the autoexposure field
            if strcmp(val,'on'), val = 1;
            else val = 0;
            end
        end
        isa.AE = val;
        if val, isa.integrationTime = 0; end
    case {'cds','correlateddoublesampling'}
        isa.CDS = val;
    case {'vignetting','pixelvignetting','sensorvignetting','bareetendue','sensorbareetendue','nomicrolensetendue'}
        % This is an array that describes the loss of light at each
        % pixel due to vignetting.  The loss of light when the microlens is
        % in place is stored in the isa.etendue entry.  The improvement of
        % the light due to the microlens is calculated from isa.etendue ./
        % isa.data.vignetting.
        isa.data.vignetting = val;

    case {'data'}
        isa.data = val;
    case {'voltage','volts'}
        isa.data.volts = val;
        %  We adjust the row and column size to match the data. The data
        %  size is the factor that determines the ISA row and column values.
        %  The row and col values are only included for those cases when
        %  there are no data computed yet.
    case {'analoggain','ag'}
        isa.analogGain = val;
    case {'analogoffset','ao'}
        isa.analogOffset = val;
    case {'dv','digitalvalue','digitalvalues'}
        isa.data.dv = val;
    case {'roi'}
        isa.roi = val;
    case {'quantization','qmethod','quantizationmethod'}
        isa = sensorSetQuantization(isa,val);

    case {'offsetfpnimage','dsnuimage'} % Dark signal non uniformity (DSNU) image
        isa.offsetFPNimage = val;
    case {'gainfpnimage','prnuimage'}   % Photo response non uniformity (PRNU) image
        isa.gainFPNimage = val;
    case {'dsnulevel','sigmaoffsetfpn','offsetfpn','offset','offsetsd','offsetnoisevalue'}
        % Units are volts
        isa.sigmaOffsetFPN = val;
        % Clear the dsnu image when the dsnu level is reset
        isa = sensorSet(isa,'dsnuimage',[]);    
    case {'prnulevel','sigmagainfpn','gainfpn','gain','gainsd','gainnoisevalue'}
        % This is stored as a percentage. Always.  This is a change from
        % the past where I tried to be clever but got into trouble.
        isa.sigmaGainFPN = val;
        % Clear the prnu image when the prnu level is reset
        isa = sensorSet(isa,'prnuimage',[]);
    case {'columnfpnparameters','columnfpn','columnfixedpatternnoise','colfpn'}
        if length(val) == 2 || isempty(val), isa.columnFPN = val;
        else error('Column fpn must be in [offset,gain] format.');
        end
    case {'colgainfpnvector','columnprnu'}
        if isempty(val), isa.colGain = val;
        elseif length(val) == sensorGet(isa,'cols'), isa.colGain = val;
        else error('Bad column gain data');
        end
    case {'coloffsetfpnvector','columndsnu'}
        if isempty(val), isa.colOffset = val;
        elseif length(val) == sensorGet(isa,'cols'), isa.colOffset = val;
        else error('Bad column offset data');
        end
    case {'shotnoiseflag'}
        isa.shotNoiseFlag = val;

    case {'ngridsamples','pixelsamples','nsamplesperpixel','npixelsamplesforcomputing'}
        isa.samplesPerPixel = val;
    case {'colorfilterarray','cfa'}
        isa.cfa = val;
    case {'cfapattern','pattern','filterorder'}
        isa.cfa.pattern = val;
        %     case {'cfaunitblock','unitblock'}
        %         isa.cfa.unitBlock = val;

    case {'pixel','imagesensorarraypixel'}
        isa.pixel = val;

        % Microlens related
    case {'microlens','ml'}
        isa.ml = val;
    case {'etendue','sensoretendue','imagesensorarrayetendue'}
        % The size of etendue should match the size of the sensor array. It
        % is computed using the chief ray angle.
        isa.etendue = val;
    case {'microlensoffset','mloffset','microlensoffsetmicrons'}
        % This is the offset of the microlens in microns as a function of
        % number of pixels from the center pixel.  The unit is microns
        % because the pixels are usually in microns.  This may have to
        % change to meters at some point for consistency.
        isa.mlOffset = val;

    case {'consistency','sensorconsistency'}
        isa.consistency = val;
    case {'sensorcompute','sensorcomputemethod'}
        isa.sensorComputeMethod = val;

    case {'mccrecthandles'}
        % These are handles to the squares on the MCC selection regions
        % see ieMacbethSelect.  If we over-write them, we first delete the
        % existing ones.
        if checkfields(isa,'mccRectHandles')
            if ~isempty(isa.mccRectHandles),
                try delete(isa.mccRectHandles(:));
                catch
                end
            end
        end
        isa.mccRectHandles = val;
    otherwise
        error('Unknown parameter.');
end

return

%---------------------
function isa = sensorSetQuantization(isa,qMethod)
%
%  isa = sensorSetQuantization(isa,qMethod)
%
% Set the quantization method and bit count for an image sensor array.
%
% Examples
%   isa = sensorSetQuantization(isa,'analog')
%   isa = sensorSetQuantization(isa,'10 bit')

if ~exist('qMethod','var') || isempty(qMethod), qMethod = 'analog'; end

% Read the entries in the conttents and do the right thing to the
% quantization entries.
switch lower(qMethod)
    case 'analog'
        isa.quantization.bits = [];
        isa.quantization.method = 'analog';
    case '4 bit'
        isa.quantization.bits = 4;
        isa.quantization.method = 'linear';
    case '8 bit'
        isa.quantization.bits = 8;
        isa.quantization.method = 'linear';
    case '10 bit'
        isa.quantization.bits = 10;
        isa.quantization.method = 'linear';
    case '12 bit'
        isa.quantization.bits = 12;
        isa.quantization.method = 'linear';
    case 'sqrt'
        isa.quantization.bits = 8;
        isa.quantization.method = 'sqrt';
    case 'log'
        isa.quantization.bits = 8;
        isa.quantization.method = 'log';
    otherwise
        warning('Unknown quantization method.');
end

return;