function ISA = sensorCreate(isaName,PIXEL,varargin)
%Create an image sensor array structure
%
%      ISA = sensorCreate(isaName,[PIXEL])
%
% The sensor array uses a pixel definition that can be specified in the
% parameter PIXEL. If this is not passed in, a default PIXEL is created
% and returned.  
%
% Types of sensors
%  Bayer RGB combinations
%      {'bayer-grbg'}
%      {'bayer-rggb'}
%      {'bayer-bggr'}
%      {'bayer-gbrg'}
%  Bayer CMY combinations
%      {'bayer (ycmy)'}
%      {'bayer (cyym)'}
%  
%      {'monochrome'}
%
% Multiple channel possibilities are created this way
%      {'fourcolor-grbc'}
%      {'custom'}
%
% Hmm.  Some kind of human thing.  More notes later
%      {'human'}
%         
% See also:  sensorCreateIdeal
%
% Examples
%  ISA = sensorCreate;
%  ISA = sensorCreate('default'); 
%  ISA = sensorCreate('bayer (ycmy)');
%  ISA = sensorCreate('bayer (rggb)');
%  ISA = sensorCreate('Monochrome');
%
%  cone = pixelCreate('humancone'); ISA = sensorCreate('Monochrome',cone);
%  ISA = sensorCreate('human');
%  ISA = sensorCreate('fourColor');
%
%  sensor = sensorCreate; pixel = sensorGet(sensor,'pixel');
%  filterOrder = [1 2 3; 4 5 2; 3 1 4];
%  filterFile = fullfile(isetRootPath,'data','sensor','colorfilters','sixChannel.mat');
%  sensor = sensorCreate('custom',pixel,filterOrder,filterFile)
%
% Copyright ImagEval Consultants, LLC, 2005

% TODO
%  Bring sensorCreateIdeal into here?

if ieNotDefined('isaName'), isaName = 'default'; end

ISA.name = [];
ISA.type = 'ISA';

% Make sure a pixel is defined.
if ieNotDefined('PIXEL')
    PIXEL = pixelCreate('default',0);
    ISA = sensorSet(ISA,'pixel',PIXEL);
    ISA = sensorSet(ISA,'size',sensorFormats('qqcif'));
else
    ISA = sensorSet(ISA,'pixel',PIXEL);
end

% The sensor should always inherit the spectrum of the pixel.  Probably
% there should only be one spectrum here, not one for pixel and ISA.
ISA = sensorSet(ISA,'spectrum',pixelGet(PIXEL,'spectrum'));

% These should all be sensorSet calls.
ISA = sensorSet(ISA,'data',[]);

ISA = sensorSet(ISA,'sigmagainfpn',0);    % [V/A]  This is the slope of the transduction function
ISA = sensorSet(ISA,'sigmaoffsetfpn',0);  % V      This is the offset from 0 volts after reset

ISA = sensorSet(ISA,'integrationTime',0);
ISA = sensorSet(ISA,'autoexposure',0);
ISA = sensorSet(ISA,'CDS',0);

% I wonder if the default spectrum should be hyperspectral, or perhaps it
% should be inherited from the currently selected optical image?
% ISA = initDefaultSpectrum(ISA,'hyperspectral');

ISA = sensorSet(ISA,'analogGain',1);
ISA = sensorSet(ISA,'analogOffset',0);
ISA = sensorSet(ISA,'offsetFPNimage',[]);
ISA = sensorSet(ISA,'gainFPNimage',[]);
ISA = sensorSet(ISA,'gainFPNimage',[]);
ISA = sensorSet(ISA,'quantization','analog');

switch lower(isaName)
    case {'default','color','bayer','bayer (grbg)','bayer-grbg','bayergrbg'}
        filterOrder = [2,1;3,2];
        filterFile = 'rgb';
        ISA = sensorBayer(ISA,filterOrder,filterFile);
    case {'bayer (rggb)','bayer-rggb'}
        filterOrder = [1 2 ; 2 3];
        filterFile = 'rgb';
        ISA = sensorBayer(ISA,filterOrder,filterFile);
    case {'bayer (bggr)','bayer-bggr'}
        filterOrder = [3 2 ; 2 1];
        filterFile = 'rgb';
        ISA = sensorBayer(ISA,filterOrder,filterFile);
    case {'bayer (gbrg)','bayer-gbrg'}
        filterOrder = [2 3 ; 1 2];
        filterFile = 'rgb';
        ISA = sensorBayer(ISA,filterOrder,filterFile);
% added to get ground truth image
    case {'bayer (rrrr)','bayer-rrrr'}
        filterOrder = [1 1 ; 1 1];
        filterFile = 'rgb';
        ISA = sensorBayer(ISA,filterOrder,filterFile);
    case {'bayer (gggg)','bayer-gggg'}
        filterOrder = [2 2 ; 2 2];
        filterFile = 'rgb';
        ISA = sensorBayer(ISA,filterOrder,filterFile);
    case {'bayer (bbbb)','bayer-bbbb'}
        filterOrder = [3 3 ; 3 3];
        filterFile = 'rgb';
        ISA = sensorBayer(ISA,filterOrder,filterFile);
    case {'bayer (ycmy)','bayer-ycmy'}
        filterFile = 'cym';
        filterOrder = [2,1; 3,2];
        ISA = sensorBayer(ISA,filterOrder,filterFile);
    case {'bayer (cyym)','bayer-cyym'}
        filterFile = 'cym';
        filterOrder = [1 2 ; 2 3];
        ISA = sensorBayer(ISA,filterOrder,filterFile);
    case {'custom'}  % Often used for multiple channel
        % sensorCreate('custom',pixelCreate,filterFile,filterOrder);
        if length(varargin) >= 1, filterOrder = varargin{1};
        else  % Must read it here
        end
        if length(varargin) >= 2, filterFile = varargin{2};
        else % Must read it here
        end
        ISA = sensorCustom(ISA,filterOrder,filterFile);
    case 'monochrome'
        filterFile = 'Monochrome';
        ISA = sensorMonochrome(ISA,filterFile);
    case 'human'
        filterFile = 'stockmanAbs';
        filterOrder = [2,1; 3,2];
        ISA = sensorHuman(ISA,filterOrder,filterFile);
        ISA = sensorSet(ISA,'pixel',pixelCreate('human',0));
        ISA =sensorSet(ISA,'name','human');
    otherwise
        error('Unknown ISA type');
end

% Put in a default infrared filter.  All ones.
ISA = sensorSet(ISA,'irfilter',ones(sensorGet(ISA,'nwave'),1));
ISA = sensorSet(ISA,'mccRectHandles',[]);

return;

%-----------------------------
function ISA = sensorBayer(ISA,filterOrder,filterFile)
%
%   Create a default image sensor array structure.

ISA = sensorSet(ISA,'name',sprintf('bayer-%.0f',vcCountObjects('ISA')));
ISA = sensorSet(ISA,'filterorder',filterOrder);

% Read in a default set of filter spectra
[filterSpectra,filterNames] = sensorColorFilters(ISA,filterFile);
ISA = sensorSet(ISA,'filterspectra',filterSpectra);
ISA = sensorSet(ISA,'filternames',filterNames);

return;

%-----------------------------
function ISA = sensorHuman(ISA,filterOrder,filterFile)
%
%   Create a default image sensor array structure.

ISA = sensorSet(ISA,'name',sprintf('human-%.0f',vcCountObjects('ISA')));
ISA = sensorSet(ISA,'cfaPattern',filterOrder);

% Read in a default set of filter spectra
[filterSpectra,filterNames] = sensorColorFilters(ISA,filterFile);
ISA = sensorSet(ISA,'filterSpectra',filterSpectra);
ISA = sensorSet(ISA,'filterNames',filterNames);

return;

%-----------------------------
function ISA = sensorCustom(ISA,filterOrder,filterFile)
%
%  Set up a sensor with multiple color filters.
%

ISA = sensorSet(ISA,'name',sprintf('custom-%.0f',vcCountObjects('ISA')));

ISA = sensorSet(ISA,'cfaPattern',filterOrder);      

[filterSpectra,filterNames] = sensorColorFilters(ISA,filterFile);
ISA = sensorSet(ISA,'filterSpectra',filterSpectra);
ISA = sensorSet(ISA,'filterNames',filterNames);

return;

%-----------------------------
function ISA = sensorMonochrome(ISA,filterFile)
%
%   Create a default monochrome image sensor array structure.
%

ISA = sensorSet(ISA,'name',sprintf('monochrome-%.0f', vcCountObjects('ISA')));

[filterSpectra,filterNames] = sensorColorFilters(ISA,filterFile);
ISA = sensorSet(ISA,'filterSpectra',filterSpectra);
ISA = sensorSet(ISA,'filterNames',filterNames);

ISA = sensorSet(ISA,'cfaPattern',1);      % 'bayer','monochromatic','triangle'
% cOrder = sensorGet(ISA,'colorOrder');
% ISA = sensorSet(ISA,'unitBlock',sensorUnitBlock(ISA,cOrder));

return;

%---------------------------------------------
function [filterSpectra, filterNames] = sensorColorFilters(ISA,filterFile)
%
%   [filterSpectra, filterNames] = sensorColorFilters(ISA,filterType)
%
%    Assign color filter spectral transmissivities.

wave = sensorGet(ISA,'wave');
nWave = sensorGet(ISA,'nwave');

switch lower(filterFile)
    case 'xyz'
        fname = fullfile(isetRootPath,'data','human','XYZ');
    case 'rgb'
        fname = fullfile(isetRootPath,'data','sensor','RGB.mat');
    case 'monochrome'
        % Should update this monochrome sensor default to a more plausible
        % PD spectral responsivity.
        filterSpectra = ones(nWave,1);
        filterNames = {'w'};
        return;
    case 'cym'
        fname = fullfile(isetRootPath,'data','sensor','cym.mat');
    case {'grbc'}
        fname = fullfile(isetRootPath,'data','sensor','GRBC.mat');
    case 'stockmanabs'
        fname = fullfile(isetRootPath,'data','human','stockman.mat');
    otherwise
        if exist(filterFile,'file'), fname = filterFile;
        else                         fname = vcSelectDataFile('sensor','r');
        end
end

[filterSpectra,filterNames] = ieReadColorFilter(wave,fname);

% filterSpectra = vcReadSpectra(fname, wave);
% load(fname,'filterNames');

return;

