% This file collects the MSE and sCIELAB values for each setting
% in the Denoise-Demosaic pipeline and saves the data in a CSV file.
clear;clc;close all;

addpath scielab

% algorithms
DEMOSAIC{1} = 'bilinear';
DEMOSAIC{2} = 'homogeneity';
DEMOSAIC{3} = 'pocs';
DEMOSAIC{4} = 'frequency';

DENOISE{1} = 'simple';      dn1p1 = [0 0 0 0];  dn1p2 = [0 0 0 0];       % parameters for each noise level
DENOISE{2} = 'bm3d';        dn2p1 = [1 3 9 10]; dn2p2 = [0 0 0 0];       % p1 - sigma, p2 - don't care
DENOISE{3} = 'bilateral';   dn3p1 = [1 1 1 1];  dn3p2 = [0.05 0.05 0.10 0.20];     % p1 - sigma spatial, p2 - sigma range
DENOISE{4} = 'bls_gsm';     dn4p1 = [2/255 5/255 10/255 15/255]; dn4p2 = [0 0 0 0];   % p1 - sigma, p2 - don't care

dnParam1 = [ dn1p1; dn2p1; dn3p1; dn4p1 ];
dnParam2 = [ dn1p2; dn2p2; dn3p2; dn4p2 ];

% display settings
displayPlots = true;

% parameter settings
% pipeline type
plType = 'DN2DM';

M = zeros(7*4*3*4, 13);

count = 0;
% image number
for imageNo = 1:7
fc = 0;
% select noise level
for noiseLevel = 1:4
% select DM algorithm
for dm = 1:3
% select DN algorithm
for dn = 1:4
    
count = count+1;
fc = fc+1;
disp(sprintf('count = %d', count));

% save settings
M(fc, 1) = imageNo; 
M(fc, 2) = noiseLevel;
M(fc, 3) = dm;
M(fc, 4) = dn;

% load data
[imData fileName] = loadImage(imageNo);

% true data
imTRUE = imData{1};  % ref image
imRAW = mosaicRGB(imData{1});   % clean raw image

% load noisy image
imNoDn = imData{5+noiseLevel};
imPro = imData{5+noiseLevel};
imPar = zeros(size(imTRUE));

%% Stage 1 Denoise
disp('Stage 1 - Denoise');
tic;

% decompose channels
imParRgb = decomposeRAW2(imRAW);
imProRgb = decomposeRAW2(imPro);

% denoise each channel
if(dn == 2)
    imPar = applyDenoiseSingle(imRAW, DENOISE{dn}, dnParam1(dn, noiseLevel), dnParam2(dn, noiseLevel));
    imPro = applyDenoiseSingle(imPro, DENOISE{dn}, dnParam1(dn, noiseLevel), dnParam2(dn, noiseLevel));
else
    for c = 1:3
        scale = 1;
        if(c == 2 && dn ~= 3) scale = 2; end
        imParRgb{c} = applyDenoiseSingle(imParRgb{c}, DENOISE{dn}, scale*dnParam1(dn, noiseLevel), scale*dnParam2(dn, noiseLevel));
        imProRgb{c} = applyDenoiseSingle(imProRgb{c}, DENOISE{dn}, scale*dnParam1(dn, noiseLevel), scale*dnParam2(dn, noiseLevel));
    end

    % compose denoised image
    imPar = composeRGB(imParRgb{1}, imParRgb{2}, imParRgb{3});
    imPro = composeRGB(imProRgb{1}, imProRgb{2}, imProRgb{3});
end

% compute errors
% 1. Clean image errors - errors of denoise itself
mse = evaluateQuality(imRAW, imPar, 'mse');
M(fc, 5) = mse;

% 2. True image errors - deviation from true image
mse = evaluateQuality(imRAW, imPro, 'mse');
M(fc, 6) = mse;

% 3. No denoise image errors - errors from denoise
% mse = evaluateQuality(imNoDn, imPro, 'mse');
% M(count, 7) = mse;

% 3. Parallel image errors - errors from denoise
mse = evaluateQuality(imPar, imPro, 'mse');
M(fc, 7) = mse;

% display images
if(displayPlots)
    % look at each channel
    
    imNoDnRgb = decomposeRAW2(imNoDn);
    imProRgb = decomposeRAW2(imPro);
    
    figure('Name', ['[Stage 1] - ' fileName]);

    c= 1;
    subplot(231);
    displayRGB(imNoDnRgb{c}); title('Noisy Image - Red');

    subplot(234);
    displayRGB(imProRgb{c}); title('Denoised Image - Red');

    c= 2;
    subplot(232);
    displayRGB(imNoDnRgb{c}); title('Noisy Image - Green');
    
    subplot(235);
    displayRGB(imProRgb{c}); title('Processed Image - Green');
    
    c= 3;
    subplot(233);
    displayRGB(imNoDnRgb{c}); title('Noisy Image Image - Blue');
    
    subplot(236);
    displayRGB(imProRgb{c}); title('Processed Image - Blue');
    
end
toc;

%% Stage 2 - Demosaic %%
disp('Stage 2 - Demosaic');
tic;

% demosaic images
if(displayPlots) imNoDn = applyDemosaic(imNoDn, DEMOSAIC{dm}); end % no denoise image
imPar = applyDemosaic(imPar, DEMOSAIC{dm}); % parallel processed clean image
imPro = applyDemosaic(imPro, DEMOSAIC{dm}); % processed image

% calculate errors
% 1. Clean image errors
mse = evaluateQuality(imTRUE, imPar, 'mse');
scielab = evaluateQuality(imTRUE, imPar, 'scielab');
M(fc, 8) = mse;
M(fc, 9) = scielab;

% 2. True image errors
mse = evaluateQuality(imTRUE, imPro, 'mse');
scielab = evaluateQuality(imTRUE, imPro, 'scielab');
M(fc, 10) = mse;
M(fc, 11) = scielab;

% 3. Parallel image errors
mse = evaluateQuality(imPar, imPro, 'mse');
scielab = evaluateQuality(imPar, imPro, 'scielab');
M(fc, 12) = mse;
M(fc, 13) = scielab;

% display images
if(displayPlots)
    figure('Name', ['[Stage 2] - ' fileName]);
    subplot(221); 
    displayRGB(imTRUE); title('True Image');

    subplot(222);
    displayRGB(imPar); title('Parallel Image');
    
    subplot(223); 
    displayRGB(imNoDn); title('No Denoise Image');

    subplot(224); 
    displayRGB(imPro); title('Processed Image');
end
toc;

end % dn
end % dm
end % noiseLevel

% write to file
csvwrite([plType '_' fileName '.csv'], M);
disp(['saved to file: ' plType '_' fileName '.csv']);

end % imageNo