%
% displays the error of the image compared to a reference image
%
% parameters
% im - the image to compare
% imRef - the reference(ground truth) image
% type - the type of error to view
% param - additional parameters
%
% the type of error displays
% 'errorImage'  - displays the error image
% 'rgb3D'       - displays the RGB of errors in 3D space
% 'rgbPlane'    - displays the projection of 'rgb3D' on to the RG/GB/BR
%                 surface
% 'histogram'   - displays the histogram of errors
% 'edImage'     - displays error difference image
% 'edHistogram' - displays error difference histogram
% 'viPlot'      - displays the Intensity-Variance plot
% 'autocorrelation' - displays the autocorrelation
% 'autocorPatch'    - displays the autocorrelation of patches for faster results 
%
% TODO: handle single band images
%
function displayError(im, imRef, type, param)

addpath errorPlots

% get dimensions
[h w] = size(im(:, :, 1));
l = w*h;

% get error image
imErr = im-imRef;

switch type
    case 'errorImage'   % displays the error image
        displayRGB(imErr);
        
    case 'rgb3D'    % displays the RGB of errors in 3D space
        displayRgb3D(im, imRef);

    case 'rgbPlane' % displays the projection of 'rgb3D' on to the RG/GB/BR
        displayRgbPlane(im, imRef);
        
    case 'histogram'    % displays the histogram of errors
        displayHistogram(imErr);

    case 'edImage'  % displays error difference image
        imErrDiff = createErrDiffImage(imErr);
        displayRGB(imErrDiff);
        
    case 'edHistogram'    % displays error difference histogram
        imErrDiff = createErrDiffImage(imErr);
        displayHistogram(imErrDiff);
        
    case 'viPlot'   % displays the Intensity-Variance plot
        displayViPlot(im, imRef);
        
    case 'autocorrelation' % displays the autocorrelation
        displayAutoCor(im, imRef);
        
    case 'autocorPatch' % displays the autocorrelation
        if exist('param') == 0, param = 64; end
        
        displayAutoCorPatch(im, imRef, param);
end

%
% Creates an error difference image from an error image.
%
function imErrDiff = createErrDiffImage(imErr)

r = squeeze(imErr(:, :, 1));
r = conv2(r, [0 -1 0; -1 4 -1; 0 -1 0]);
g = squeeze(imErr(:, :, 2));
g = conv2(g, [0 -1 0; -1 4 -1; 0 -1 0]);
b =  squeeze(imErr(:, :, 3));
b = conv2(b, [0 -1 0; -1 4 -1; 0 -1 0]);

[hh ww] = size(r);
ll = hh*ww;
imErrDiff = zeros(hh, ww, 3);
imErrDiff(:, :, 1) = r;
imErrDiff(:, :, 2) = g;
imErrDiff(:, :, 3) = b;