/*
	FreeRTOS.org V4.1.3 - Copyright (C) 2003-2006 Richard Barry.

	This file is part of the FreeRTOS.org distribution.

	FreeRTOS.org is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	FreeRTOS.org is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with FreeRTOS.org; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

	A special exception to the GPL can be applied should you wish to distribute
	a combined work that includes FreeRTOS.org, without being obliged to provide
	the source code for any proprietary components.  See the licensing section 
	of http://www.FreeRTOS.org for full details of how and when the exception
	can be applied.

	***************************************************************************
	See http://www.FreeRTOS.org for documentation, latest information, license 
	and contact details.  Please ensure to read the configuration and relevant 
	port sections of the online documentation.
	***************************************************************************
*/

#define FOSC 16000000

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/eeprom.h>

#ifdef GCC_MEGA_AVR
	/* EEPROM routines used only with the WinAVR compiler. */
	#include <avr/eeprom.h> 
#endif

/* Scheduler include files. */
#include "FreeRTOS.h"
#include "task.h"
#include "croutine.h"
#include "queue.h"
#include "twi.h"
#include "lm75.h"


void vWatchTemp ( void *pvParameters);
void vFlashOK( void *pvParameters );
void vMenu( void *pvParameters );
void MenuPrintHelp( void );
void MenuPrintConfig ( void );
void ShootIt( void );
void CameraFire( void );
void CameraUnFire( void );
void LEDOn( unsigned char led );
void LEDOff( unsigned char led );

void EasterEgg1( void );
void EasterEgg2( void );
void EasterEgg3( void );



void USART_init( double baud, char databits, char pairitytype, char stopbits );
void USART0_txchar( char data );
void USART0_rxchar( char *data );
void USART0_txline( char *line );
void vApplicationIdleHook( void );


//Global Variables
xQueueHandle UARTRxQueue;
xQueueHandle UARTTxQueue;
unsigned long MirrorLockupTime=0;
unsigned short DelayTime=400;
unsigned short EnabledColors=0b111111111;
unsigned char ShotsPerColor=1;
unsigned short repeat=1;

/*-----------------------------------------------------------*/

portSHORT main( void )
{
	// Set up the GPIO Directions
	DDRH=0x38; // Cyan, Green, Amber (LSB->MSB)
	DDRL=0x38; //Red, 880, 940 (LSB->MSB)
	DDRE=0xFA; //USART and UV, Blue0, Blue1
	DDRK=0x0f; //Status LEDs


	//Start with everything off
	PORTE=0x00;
	PORTH=0x00;
	PORTK=0x00;
	PORTL=0x00;

	//Initialize the USART
	USART_init(115200, 8, 0, 1);

	//Initialize the queues for the USART
	UARTRxQueue = xQueueCreate( 5, sizeof( unsigned char ));
	UARTTxQueue = xQueueCreate( 5, sizeof( unsigned char ));
	
	vMonitorTemp(0);


	// Task to indicate that the system is running (ie "cycle" led)
	xTaskCreate( vFlashOK, "FlashOK", 300, NULL, 1, NULL );

	// Task to indicate that the system is running (ie "cycle" led)
	xTaskCreate( vMenu, "Menu", 600, NULL, 1, NULL );

	// Task to indicate that the system is running (ie "cycle" led)
	xTaskCreate( vWatchTemp, "WatchTemp", 600, NULL, 2, NULL );

	
	/* In this port, to use preemptive scheduler define configUSE_PREEMPTION 
	as 1 in FreeRTOSConfig.h.  To use the cooperative scheduler define 
	configUSE_PREEMPTION as 0. */
	vTaskStartScheduler();

	// PORTG=0xff;
	while (1) {
		asm volatile ("nop");
	}
	return 0;
}


void vFlashOK (void *pvParameters)
{
	/* The parameters are not used. */
	( void ) pvParameters;
	
	for(;;) 
	{
		PORTK|=0x01; 		//OK LED On
		vTaskDelay( 500 );
		PORTK&=0xfe;		//OK LED Off
		vTaskDelay( 500 );
	}
}

void vWatchTemp ( void *pvParameters)
{
	//THIS TASK MUST BE THE HIGHEST PRIORITY TO PREVENT DAMAGE TO THE UNIT
	signed char temp;
	unsigned char old_e, old_h, old_l, old_k;
	( void ) pvParameters;
	
	temp = read_temp1();
	for(;;)
	{
		if (temp > 50) // lock out!
		{
//			vTaskSuspendAll();
			


			USART0_txline("\r\nWARNING: UNIT TOO HOT.\r\nStandby for cool down, will resume automatically when safe...\0");
			vTaskDelay(1000); //give our selves a bit to get the warning message out before lockdown :)
			taskENTER_CRITICAL();
			old_e = PORTE; //Save LED State
			old_h = PORTH;
			old_l = PORTL;
			old_k = PORTK;

			PORTE &= 0xff-0x38;	//Turn off the LEDs
			PORTH &= 0xff-0x38;
			PORTL &= 0xff-0x38;

			PORTK = 0x08; //Turn on the warning LED

			while (temp >45)
			{
				temp = read_temp1();
			}

			PORTK = old_k; //Turn off warning LED

			PORTE = old_e; // Restore LED State
			PORTH = old_h;
			PORTL = old_l;

//			xTaskResumeAll();
			USART0_txline("Resuming.\r\n");
			taskEXIT_CRITICAL();
		}
		vTaskDelay(1000);
		temp = read_temp1();
	}

}

void vMenu (void *pvParameters)
{
	unsigned char input[10];
	char buffer[20];
	unsigned char i;
	long buff_num;

	/* The parameters are not used. */
	( void ) pvParameters;
	
	USART0_txline("\r\n\r\n\r\n\r\n\r\n\r\n\r\n\r\n\r\n\r\n\r\n\r\n");
	USART0_txline("\r\n\r\n\r\n\r\n\r\n\r\n\r\n\r\n\r\n\r\n\r\n\r\n");
	USART0_txline("Multispectral Illuminator\r\nFirmware Version 1.0\r\n\0");
	MenuPrintHelp();
	for(;;) 
	{
		USART0_txchar('\r');
		USART0_txchar('\n');
		USART0_txchar('>');
		i=0;
		USART0_rxchar( input );	// wait for user input
		USART0_txchar( input[0] ); //Echo what the user types
		while ((input[i] != '\r') && (i<9))
		{
			i++;
			USART0_rxchar( input+i ); // get the next character of user input
			USART0_txchar( input[i] );
		}
		input[i]='\0'; //Null terminate the string
		USART0_txchar('\n');

		switch (input[0])				// run respective functions
		{								
			case '?':
			{
				MenuPrintHelp();
				break;
			}
			case 'p':
			{
				MenuPrintConfig();
				break;
			}
			case 'g':
			{
				for (i=0;i<repeat;i++)
					ShootIt();
				break;
			}
			case 'm':
			{
				buff_num=atoi(input+1);
				MirrorLockupTime=buff_num;
				USART0_txline("OK\0");
				break;
			}
			case 'w':
			{
				buff_num=atoi(input+1);
				if (buff_num < 5) {
					USART0_txline("Error: Minimum time between shots is 5ms.\0");
				} else {
					DelayTime=buff_num;
					USART0_txline("OK\0");
				}
				break;
			}
			case '#':
			{
				buff_num=atoi(input+1);
				if (buff_num > 255) {
					USART0_txline("Error: Maximum shots per color is 255.\0");
				} else if (buff_num <1) {
					USART0_txline("Error: Minimum number of shots per color is 1.\0");
				} else {
					ShotsPerColor=buff_num;
					USART0_txline("OK\0");
				}
				break;
			}

			case 'E':
			case 'e':
			{
				buff_num=atoi(input+1);
				if ((buff_num>9) || (buff_num<1))
				{
					USART0_txline("Error: Channel number must be between 1 and 9 inclusive.\0");
				} else {
					EnabledColors |= (1<<(buff_num-1));
					USART0_txline("OK\0");
				}
				break;
			}
			case 'D':
			case 'd':
			{
				buff_num=atoi(input+1);
				if ((buff_num>9) || (buff_num<1))
				{
					USART0_txline("Error: Channel number must be between 1 and 9 inclusive.\0");
				} else {
					EnabledColors &= (0b111111111-(1<<(buff_num-1)));
					USART0_txline("OK\0");
				}
				break;
			}
			case 'r':
			case 'R':
			{
				buff_num=atoi(input+1);
				if (buff_num<1)
				{
					USART0_txline("Error: Sequence must execute at least once.\0");
				} else {
					repeat=buff_num;
					USART0_txline("OK\0");
				}
				break;
			}
			case 'T':
			case 't':
			{
				USART0_txline("\tCurrent Hot Plate Temperature: \0");
				itoa((signed char)read_temp1(),buffer,10);
				USART0_txline(buffer);
				USART0_txchar(167);
				USART0_txline("C\r\n\0");
				USART0_txline("\tCurrent PCB Exterior Temperature: \0");
				itoa((signed char)read_temp0(),buffer,10);
				USART0_txline(buffer);
				USART0_txchar(167);
				USART0_txline("C\r\n\0");
				break;
			}
			case '1':
			{
				EasterEgg1();
				break;
			}
			case '2':
			{
				EasterEgg2();
				break;
			}
			case '3':
			{
				EasterEgg3();
				break;
			}
			default:
			{
				USART0_txline("Error: Unknown Command. Press ? for help.\0");
				break;
			}
		}


	}
}

void MenuPrintHelp ( void )
{
	USART0_txline("\r\nCommand Help:\r\n\0");
	USART0_txline("-------------\r\n\0");
	USART0_txline("?\tShow this help menu\r\n\0");
	USART0_txline("p\tPrint configuration\r\n\0");
	USART0_txline("g\tShoot sequence of pictures as configured\r\n\0");
	USART0_txline("mn\tMirror lockup delay - 50mS, or 0 to disable (n us delay in ms) [0]\r\n\0");
	USART0_txline("wn\tSet wait time for each camera fire (n is time in ms) [400]\r\n\0");
	USART0_txline("#n\tNumber of camera shots per color (n is number of shots) [1]\r\n\0");
	USART0_txline("En\tEnable color n for sequence\r\n\0");
	USART0_txline("Dn\tDisable color n for sequence\r\n\0");
	USART0_txline("rn\tRepeat the sequence n times\r\n\0");
	USART0_txline("t\tDisplay Temperatures\r\n\0");

}

void MenuPrintConfig ( void )
{
	char buffer[20];

	//Configuration Header
	USART0_txline("\r\nCurrent Configuration:\r\n\0");
	USART0_txline("----------------------\r\n\0");

	//Mirror Lockup Information
	if (MirrorLockupTime==0)
	{
		USART0_txline("\tMirror Lockup: Disabled\r\n\0");
	} else {
		USART0_txline("\tMirror Lockup Delay: \0");
		itoa(MirrorLockupTime,buffer,10);
		USART0_txline(buffer);
		USART0_txline("ms\r\n\0");
	}

	//Delay between shots
	USART0_txline("\tTime between camera shots: \0");
	itoa(DelayTime,buffer,10);	
	USART0_txline(buffer);
	USART0_txline("ms\r\n\0");

	//Shots per color
	USART0_txline("\tShots per color: \0");
	itoa(ShotsPerColor,buffer,10);	
	USART0_txline(buffer);
	USART0_txline("\r\n\0");

	//Repeat Cycle n Times
	USART0_txline("\tRepeat sequence \0");
	itoa(repeat,buffer,10);	
	USART0_txline(buffer);
	USART0_txline(" time(s).\r\n\0");

	//Enabled Colors
	USART0_txline("\tEnabled Colors:\r\n\0");

	if ((EnabledColors & 0b1) == 1)
		USART0_txline("\t\t(1)   UV 430nm\r\n\0");

	if ((EnabledColors & 0b110) == 0b110)
	{
		USART0_txline("\t\t(2+3) Royal Blue, 2x\r\n\0");
	} else {
		if ((EnabledColors & 0b10) == 0b10)
			USART0_txline("\t\t(2)   Royal Blue, 1x\r\n\0");
		if ((EnabledColors & 0b100) == 0b100)
			USART0_txline("\t\t(3)   Royal Blue, 1x\r\n\0");
	}

	if ((EnabledColors & 0b1000) == 0b1000)
		USART0_txline("\t\t(4)   Cyan\r\n\0");

	if ((EnabledColors & 0b10000) == 0b10000)
		USART0_txline("\t\t(5)   Green\r\n\0");

	if ((EnabledColors & 0b100000) == 0b100000)
		USART0_txline("\t\t(6)   Amber\r\n\0");

	if ((EnabledColors & 0b1000000) == 0b1000000)
		USART0_txline("\t\t(7)   Red\r\n\0");

	if ((EnabledColors & 0b10000000) == 0b10000000)
		USART0_txline("\t\t(8)   IR 880nm\r\n\0");

	if ((EnabledColors & 0b100000000) == 0b100000000)
		USART0_txline("\t\t(9)   IR 940nm\r\n\0");

	//Current Temperature
	USART0_txline("\tCurrent Hot Plate Temperature: \0");
	itoa((signed char)read_temp1(),buffer,10);
	USART0_txline(buffer);
	USART0_txchar(167);
	USART0_txline("C\r\n\0");
	USART0_txline("\tCurrent PCB Exterior Temperature: \0");
	itoa((signed char)read_temp0(),buffer,10);
	USART0_txline(buffer);
	USART0_txchar(167);
	USART0_txline("C\r\n\0");
}

void ShootIt( void )
{
	unsigned char pos=0;
	unsigned char shot;
	PORTK |= 0x04; //Show that we're busy
	for (pos=0; pos<9; pos++)
	{
	for (shot=0; shot<ShotsPerColor; shot++)
	{
//if (((pos == 2) && ((EnabledColors & 0b110) == 0b110))
//{
//blah
//} else {
		if ((EnabledColors & (1<<pos)) == (unsigned short)(1<<pos))
		{
			if (!(((EnabledColors & 0b110) == 0b110) && (pos == 2)))	//Don't do blue again if both are selected
			{
				if (MirrorLockupTime==0)
				{
					LEDOn(pos);
					CameraFire();
					vTaskDelay( DelayTime );
					CameraUnFire();
					LEDOff(pos);
				} else {
					CameraFire();
					vTaskDelay(25);	//May need to adjust these based upon the fastest "button press" that the camera can detect
					CameraUnFire();
					vTaskDelay(25); //ditto here.
					vTaskDelay(MirrorLockupTime);
					LEDOn(pos);
					CameraFire();
					vTaskDelay( DelayTime );
					CameraUnFire();
					LEDOff(pos);
				}
				vTaskDelay(10); //Brief delay to allow the shutter button to un-click before the next photo
			}
		}
	}
	}
	PORTK &= 0xff-0x04; //No longer busy
}




void CameraFire( void )
{
	PORTG |= 0x03;
}
void CameraUnFire( void )
{
	PORTG &= 0xff-0x03;
}

void LEDOn( unsigned char led )
{
	switch (led)
	{
		case 0:
		{
			PORTE = 0x08;
			break;
		}
		case 1:
		{
			if ((EnabledColors & 0b100) == 0b100)
			{
				PORTE = 0x30;
			} else {
				PORTE = 0x10;
			}
			break;
		}
		case 2:
		{
			PORTE = 0x20;
			break;
		}
		case 3:
		{
			PORTH = 0x08;
			break;
		}
		case 4:
		{
			PORTH = 0x10;
			break;
		}
		case 5:
		{
			PORTH = 0x20;
			break;
		}
		case 6:
		{
			PORTL = 0x08;
			break;
		}
		case 7:
		{
			PORTL = 0x10;
			PORTK |= 0x02; //Turn on the 2nd green LED to indicate IR is on
			break;
		}
		case 8:
		{
			PORTL = 0x20;
			PORTK |= 0x02; //Turn on the 2nd green LED to indicate IR is on
			break;
		}
	}
}

void LEDOff( unsigned char led )
{
	switch (led)
	{
		case 0:
		{
			PORTE &= 0xff - 0x08;
			break;
		}
		case 1:
		{
			if ((EnabledColors & 0b100) == 0b100)
			{
				PORTE &= 0xff - 0x30;
			} else {
				PORTE &= 0xff - 0x10;
			}
			break;
		}
		case 2:
		{
			PORTE &= 0xff - 0x20;
			break;
		}
		case 3:
		{
			PORTH &= 0xff - 0x08;
			break;
		}
		case 4:
		{
			PORTH &= 0xff - 0x10;
			break;
		}
		case 5:
		{
			PORTH &= 0xff - 0x20;
			break;
		}
		case 6:
		{
			PORTL &= 0xff - 0x08;
			break;
		}
		case 7:
		{
			PORTL &= 0xff - 0x10;
			PORTK &= 0xff - 0x02; //Turn off the 2nd green LED to indicate IR is on
			break;
		}
		case 8:
		{
			PORTL &= 0xff - 0x20;
			PORTK &= 0xff - 0x02; //Turn off the 2nd green LED to indicate IR is on
			break;
		}
	}
}


void EasterEgg1( void ) //Looks like you're being pulled over!
{
	int i,j;

	for (i=0; i<10; i++)
	{
		for (j=0; j<8; j++)
		{
			LEDOn(1);
			vTaskDelay(40);
			LEDOff(1);
			vTaskDelay(30);
		}
		vTaskDelay(30);
		for (j=0; j<8; j++)
		{
			LEDOn(6);
			vTaskDelay(40);
			LEDOff(6);
			vTaskDelay(30);
		}
		vTaskDelay(30);
	}
}

void EasterEgg2( void )	//White Light?!
{
	int i,j;

	for (i=0; i<500; i++)
	{
			LEDOn(1);
			vTaskDelay(2);
			LEDOff(1);
			LEDOn(3);
			vTaskDelay(3);
			LEDOff(3);
			LEDOn(4);
			vTaskDelay(2);
			LEDOff(4);
			LEDOn(6);
			vTaskDelay(2);
			LEDOff(6);
	}
	LEDOff(2);
	LEDOff(4);
	LEDOff(6);
}

void EasterEgg3( void )	//Colors outside of the monochromatic
{
			LEDOn(1);		//Blue
			vTaskDelay(600);
			LEDOff(1);

			LEDOn(3);		//Cyan
			vTaskDelay(600);
			LEDOff(3);

			LEDOn(4);		//Green
			vTaskDelay(600);
			LEDOff(4);

			LEDOn(6);		//Red
			vTaskDelay(600);
			LEDOff(6);

			vTaskDelay(400); //indicate change to mixed colors

			LEDOn(2);		//Purple
			LEDOn(6);
			vTaskDelay(1000);
			LEDOff(2);
			LEDOff(6);

			LEDOn(2);		//Cyan/Light Blue
			LEDOn(4);
			vTaskDelay(1000);
			LEDOff(2);
			LEDOff(4);

			PORTH = 0x28;	//Yellow
			vTaskDelay(1000);
			PORTH = 0x00;



	LEDOff(1);
	LEDOff(2);
	LEDOff(3);
	LEDOff(4);
	LEDOff(5);
	LEDOff(6);
}


void USART0_txline(char *line) 
{
	int i = 0;
PORTK |= 0x08;
	while (line[i] != '\0')
	{
		USART0_txchar(line[i]);
		i++;
	}
PORTK &= 0xff-0x08;
}




// This function will transmit one character passed to it out through USART0.  
void USART0_txchar( char data ) 
{
	
	if ( (uxQueueMessagesWaiting( UARTTxQueue) == 0) && (UCSR0A & (1 << UDRE0)) )
	{
	//	PORTA = 0xF7;
		UDR0 = data;
	}
	else
	{
		while(xQueueSend( UARTTxQueue, ( void * ) &data, 255 ) != pdPASS);
	}
	//while ( ! (UCSR0A & (1 << UDRE0)) );	// check if the USART is ready for input
	//UDR0 = data;
}

// takes in a pointer to an initialized char* and writes the received char from USART0
void USART0_rxchar( char *data )
{
	
	while(xQueueReceive( UARTRxQueue, data, ( portTickType ) 255 ) != pdTRUE) {}
	
	//while ( !( UCSR0A & (1<<RXC0) ) );
	//*data = UDR0;

}


// ISR for USART receiving
SIGNAL(USART0_RX_vect)
{
	unsigned char ReceivedChar = UDR0; // Grab RX char from USART0 buf
	
	// send to RX Queue
	xQueueSendFromISR(UARTRxQueue, &ReceivedChar, pdFALSE);
}

// ISR for USART transmission 
SIGNAL(USART0_TX_vect)
{
	unsigned char TransmitChar; // Char to be transmitted
	portBASE_TYPE xTaskWokenByReceive = pdFALSE; // not used

	if (xQueueReceiveFromISR( UARTTxQueue, ( void * ) &TransmitChar, &xTaskWokenByReceive) == pdTRUE)
	{
		UDR0 = TransmitChar;
	}

}

void USART_init( double baud, char databits, char pairitytype, char stopbits) 
{
	// set Baud Rate
	unsigned short ubbr = FOSC/16/(baud - 1);
	
	UBRR0H = (ubbr >> 8) & 0x0F;
	UBRR0L = (ubbr & 0x00FF);

	// enable USART0 transmitter and receiver
	UCSR0B |= (1 << RXEN0)|(1 << TXEN0);

	// enable RX complete interupt enable
	UCSR0B |= (1 << RXCIE0);
	
	// enable TX complete interrupt enable
	UCSR0B |= (1 << TXCIE0);
	
	// set the USART Mode to asynchronous -- zero UMSEL00 and UMSEL01
	UCSR0C &= ~(3 << UMSEL00);

	// set frame format
	// number of data bits in the frame set by UCSZ0 2:0
	switch ( databits ) 
	{
		case	5	:
		{
			UCSR0C &= ~(7 << UCSZ00);
			break;		// 5 bit character size
		}
		case	6	:
		{
			UCSR0C &= ~(3 << UCSZ00);
			UCSR0C |= (1 << UCSZ00);
			break;		// 6 bit character size
		}
		case	7	:
		{
			UCSR0C &= ~(7 << UCSZ00);
			UCSR0C |= (1 << UCSZ01);
			break;		// 7 bit character size
		}
		case	8	:
		{
			UCSR0C &= ~(1 << UCSZ02);
			UCSR0C |= (3 << UCSZ00);
			break;		// 8 bit character size
		}
		case	9	:
		{
			UCSR0C |= (7 << UCSZ00);
			break;		// 9 bit character size
		}
		default	:
		{
			UCSR0C &= ~(1 << UCSZ02);
			UCSR0C |= (3 << UCSZ00);
			break;		// 8 bit character size

		}
	} 
		
	// set number of stop bits
	UCSR0C |= ((stopbits - 1) << USBS0);
	UCSR0C &= ~(1 << USBS0) | ((stopbits - 1) << USBS0);

	// case statement for parity bit -- 0 for none, 1 for odd, 2 for even
	// USART pairty = UPMn1:0
	// USART data bits = UCSZn2:0
	// USART stop bits = USBSn
	// USART pairity modes = UPMn1, UPMn0
	switch ( pairitytype )
	{
		case 0	:	
		{
			UCSR0C &= ~((1 << UPM01) | (1 << UPM00));
			break;		// no pairity --> UPMn1 = 0, UPMn0 = 0
		}		
		case 1	:		
		{
			UCSR0C |= (1 << UPM01) | (1 << UPM00);
			break;		// odd pairity --> UPMn1 = 1, UPMn0 = 1
		}		
		case 2	:		
		{
			UCSR0C &= ~(1 << UPM00);
			UCSR0C |= (1 << UPM01);
			break;		// even pairity --> UPMn1 = 1, UPMn0 = 0
		}	
		default	:
		{
			UCSR0C &= ~((1 << UPM01) | (1 << UPM00));
			break;		// no pairity --> UPMn1 = 0, UPMn0 = 0
		}
	}
}












/*-----------------------------------------------------------*/

void vApplicationIdleHook( void )
{
	vCoRoutineSchedule();
}

