% This is the only function that needs to be changed to make the single
% pixel camera simulation works. Modifications on lines 81 and 82. 
% Since there is a single pixel, we discard any spatial interpolation. 
% Joelle Barral, 2008



function [voltImage, dsnu, prnu] = sensorComputeImage(OI,ISA,wBar)

%Main routine for computing the sensor voltage data from the optical image
%
%  [voltImage,dsnu,prnu] = sensorComputeImage(OI,ISA,wBarHandles)
%
% Compute the sensor voltage image from the sensor parameters and the
% optical image. This is the main computational routine used to compute
% the sensor image, and it calls a variety of sub-routines that implement
% many parts of the calculation.
%
% The spatial array of volts is routined by this routine.  In addition,
% the fixed pattern offset (dsnu) and photoresponse nonuniformity (prnu) 
% can also be returned.  
%   
% If you don't want to show any waitbars, set showWaitBar to 0.  Default is 1.
%
%   COMPUTATIONAL OVERVIEW
%
%       1.  The current generated at each pixel by the signal is computed
%       (signalCurrent).  This is converted to a voltage using the cur2volt
%       parameter.
%
%       2.  The dark voltage is computed and added to the signal.
%
%       3.  Shot noise is computed for the sum of signal and dark voltage
%       signal. This noise is added.  Hence, the Poisson noise includes
%       both the uncertainty due to the signal and the uncertainty due to
%       the dark voltage.  (It is possible to turn this off by a scripting
%       command using sensorSet(sensor,'shotNoiseFlag',0)).
%
%       4.  Read noise is added.
%
%       5.  If the sensor fixed pattern noises, DSNU and PRNU  were
%       previously stored, they are added/multipled into the signal.
%       Otherwise, they are computed and stored and then combined into the
%       signal.
%
%       6.  If column FPN is selected and stored, it is retrieved and
%       combined into the signal.  If column FPN is selected but not
%       stored, it is computed and applied. Finally, if it is not selected,
%       it is not applied.
%
%   Many more notes on the calculation, including all the units are
%   embedded in the comments below. 
%
%   If the waitBar handles are provided, they are used to show progress.
%   WIthout the waitbar handles, not waitbars are shown.
%
%  Example:
%     volts = sensorComputeImage(vcGetObject('oi'),vcGetObject('ISA'));
%
% Copyright ImagEval Consultants, LLC, 2003.

if ieNotDefined('wBar'), showWaitBar = 0; else showWaitBar = 1; end
if ieNotDefined('OI'), errordlg('Optical image required.'); end
if ieNotDefined('ISA'), errordlg('Image sensor array required.'); end

q = vcConstants('q');                       %Charge/electron
pixel = sensorGet(ISA,'pixel');

% This factor converts pixel current to volts for this integration time.
% The conversion units are  
%
%   sec * (V/e) * (e/charge) = sec * V / charge = V / current. 
%
% Given the basic rule V = IR, k is effectively a measure of resistance
% that converts current into volts given the exposure duration. 
cur2volt = sensorGet(ISA,'integrationTime')*pixelGet(pixel,'conversionGain') / q;

% Calculate the signal current
if   showWaitBar, voltImage = cur2volt*signalCurrent(OI,ISA,wBar);
else %voltImage = cur2volt*signalCurrent(OI,ISA);
  % disp(' Warning: This is true only for the single-pixel camera');   
    voltImage = sum(sum(oiGet(OI,'photons')))*pixelGet(pixel,'area')*q*cur2volt;
end
ISA = sensorSet(ISA,'volts',voltImage);

% Something went wrong.  Return data empty,  including the noise images.
if isempty(voltImage),  
    dsnu = []; prnu = []; 
    return; 
end

% Add the dark current
if showWaitBar, waitbar(0.8,wBar,'SensorImage: Noise'); end
darkCurrent = pixelGet(pixel,'darkCurrentDensity');
if darkCurrent ~= 0
    % At this point the noise dark current is the same at all pixels.
    % Later, we apply the PRNU gain factor to the sum of the signal and
    % noise, so that the noise dark current effectively varies across
    % pixels.  Sam Kavusi says that this variation in gain (also called
    % PRNU) is not precisely the same for signal and noise.  But we have no
    % way to assess this for most cases, so we treat the PRNU for noise and
    % signal as the same until forced to do it otherwise.
    voltImage = voltImage + pixelGet(pixel,'darkVoltage')*sensorGet(ISA,'expTime');
    
end
ISA = sensorSet(ISA,'volts',voltImage);

% Add shot noise.  Note that you can turn off shot noise in the calculation
% by setting the shotNoiseFlag to false.  Default is true.  This flag is
% accessed only through scripts at the moment.  There is no way to turn it
% off from the user interface.
if sensorGet(ISA,'shotNoiseFlag')
    voltImage = noiseShot(ISA);
    ISA = sensorSet(ISA,'volts',voltImage);
end

% Add read noise
readNoise = pixelGet(ISA.pixel,'readnoisevolts');
if readNoise ~= 0,  voltImage = noiseRead(ISA); end
ISA = sensorSet(ISA,'volts',voltImage);

% noiseFPN combines the offset and gain (dsnu,prnu) images with
% the current voltage image to produce a noisier image.  If these images
% don't yet exist, we compute them. 
dsnu = sensorGet(ISA,'dsnuImage');
prnu = sensorGet(ISA,'prnuImage');
if isempty(dsnu) || isempty(prnu)
    [voltImage,dsnu,prnu]  = noiseFPN(ISA);
    ISA = sensorSet(ISA,'dsnuImage',dsnu);
    ISA = sensorSet(ISA,'prnuImage',prnu);
    % disp('Initiating dsnu and prnu images');
else  
    voltImage = noiseFPN(ISA);
end
ISA = sensorSet(ISA,'volts',voltImage);

% Now we check for column FPN value.  If data exist then we compute column
% FPN noise.  Otherwise, we carry on.
if isempty(sensorGet(ISA,'coloffsetfpnvector')) || isempty(sensorGet(ISA,'colgainfpnvector'))
else voltImage = noiseColumnFPN(ISA);
end

% No need for this because ISA is not returned.
% ISA = sensorSet(ISA,'volts',voltImage);

return;

