%  SimuWithOptics.m 
%  Main file of the Single-pixel camera simulation 
%  l1magic is needed (www.l1-magic.org)
%  ISET is needed (http://www.imageval.com) with the modified sensorComputeImage.m
%  The paths have to be properly replaced below. 
%
%  written by Joelle Barral, 2008
%  (c) Board of Trustees, Leland Stanford Junior University

clear all
close all

addpath '/Applications/MATLAB_R2007b/toolbox/ISET-2.0'
isetPath('/Applications/MATLAB_R2007b/toolbox/ISET-2.0')
path(path, './l1magic/Optimization');
path(path, './l1magic/Measurements');
path(path, './l1magic/Data');

DISPLAY = 0  % Switch to 1 to display the GUI
working = 0; % 0: working version, mostly default parameters
             % 1: parameters estimated for the single-pixel camera
             % Most of them are guesses since the specifications were not
             % available

n2 =  52; %12: debug, 52: simulation
n = 64;   %16: debug, 64: simulation
% We need two different numbers since random_w requires a length of a power
% of 2, and oiCompute changes the size of the input
N = n*n;
M = ceil(N/4);
C = random_w(M,N);
figure
imshow(abs(C),[])
title('Sensing matrix')

ybef = zeros(M,1);    % the M measurements before digitization
ydig = zeros(M,1); % the M digitized measurements

disp(' *** Simulating Measurements ***  ')
% Original image
photons = phantom(n2); 
scene = sceneCreate('monochrome');
scene = sceneSet(scene,'filename','Shepp Logan');
scene = sceneSet(scene,'cphotons',photons);

if (working)
   scene = sceneSet(scene,'distance',1); 
   scene = sceneAdjustLuminance(scene,100); 
else 
   scene = sceneSet(scene,'fov',20); 
   scene = sceneAdjustLuminance(scene,90); 
   scene = sceneSet(scene,'distance',0.06); 
end

if (DISPLAY) vcAddAndSelectObject(scene); sceneWindow; end

% Create an optical image and put it in the optical image window
oi1 = oiCreate; % oi1 is optical image after lens 1
optics1 = oiGet(oi1,'optics');
optics1 = opticsSet(optics1,'fnumber',4);
if (working)
    optics1 = opticsSet(optics1,'focalLength',0.004);
else
    optics1 = opticsSet(optics1,'focalLength',0.080); 
end
oi1 = oiSet(oi1,'optics',optics1);
oi1 = oiCompute(scene,oi1);

if (DISPLAY) vcAddAndSelectObject(oi1); oiWindow; end

% Take M measurements
for meas=1:M
	oiM = oiCreate; % oiM is optical image after mirror
    wave = oiGet(oiM,'wave');
	img = oiGet(oi1,'cphotons');
    img = imresize(img,[n,n]); % Just to make sure nothing nasty will happen
                               % but not optimal
	S = C(meas,:);
	S = reshape(S,[n,n]);
	sensedIMG = S.*img; % S is one row of C 
	oiM = oiSet(oiM,'cphotons',sensedIMG);
	
	if (DISPLAY) vcAddAndSelectObject(oiM); oiWindow; end 
	
    % We transform this optic image into a scene in order to feed it to
    % lens 2 (Not sure if there is a better way to do that).
    scene2 = sceneCreate('monochrome');
    scene2 = sceneSet(scene2,'filename','After DMD');
    scene2 = sceneSet(scene2,'cphotons',sensedIMG);
    if (working)
        scene2 = sceneSet(scene2,'fov',4); % in deg
    else
        scene2 = sceneSet(scene2,'fov',15); % in deg 
        scene2 = sceneSet(scene2,'distance',0.08); 
    end
    
    if (DISPLAY) vcAddAndSelectObject(scene2); sceneWindow; end
	
	oi2 = oiCreate; % oi2 is optical image after lens 2
	optics2 = oiGet(oi2,'optics');
    optics2 = opticsSet(optics2,'fnumber',4); 
    if(working)
        optics2 = opticsSet(optics2,'focalLength',0.004); 
    else
        optics2 = opticsSet(optics2,'focalLength',0.1); 
    end
	oi2 = oiCompute(scene2,oi2);

	if (DISPLAY) vcAddAndSelectObject(oi2); oiWindow; end
	
    imgtest = oiGet(oi2,'cphotons');
	% Create a monochrome sensor, and compute the voltage response
    pixel =  pixelCreate('default',0);
    % 85% geometry efficiency. 
    % Probably does not make much difference since single pixel. 
    pdSize = sqrt(pixelGet(pixel,'width')*pixelGet(pixel,'width')*0.85);
    pixel = pixelSet(pixel,'pdWidth',pdSize);
    pixel = pixelSet(pixel,'pdHeight',pdSize);
    if (working)
        pixel = pixelSet(pixel,'voltageswing',100);
    else
        pixel = pixelSet(pixel,'voltageswing',N/2); 
    end
    sensor = sensorCreate('monochrome',pixel);
	sensor = sensorSet(sensor,'rows',1);
	sensor = sensorSet(sensor,'cols',1);
    sensor = sensorSet(sensor,'quantizationmethod','12 bit');
    sensor = sensorSet(sensor,'IntegrationTime',0.101); 
	sensor = sensorCompute(sensor,oi2,0); 
    
	ybef(meas) = sensorGet(sensor,'volts'); 
    ydig(meas) = sensorGet(sensor,'digitalvalues');
	
	if (DISPLAY) vcAddAndSelectObject(sensor); sensorImageWindow; end
	
end
	

X = phantom(n); 
x = X(:);
y_ideal = C*x; 

disp(' *** Reconstructing ***  ')
C2 = pinv(C);

x_hat_dummy_bef = C2*ybef; % L2
X_dummy_bef = reshape(x_hat_dummy_bef,n,n);
x_hat_cs_bef = tveq_logbarrier(x_hat_dummy_bef, C, C2, ybef, 1e-1, 2, 1e-8, 600); 
X_cs_bef = reshape(x_hat_cs_bef, n, n);

x_hat_dummy_dig = C2*ydig; % L2
X_dummy_dig = reshape(x_hat_dummy_dig,n,n);
x_hat_cs_dig = tveq_logbarrier(x_hat_dummy_dig, C, C2, ydig, 1e-1, 2, 1e-8, 600); 
X_cs_dig = reshape(x_hat_cs_dig, n, n);

figure        
subplot(1,3,1)
imshow(X_dummy_bef,[])
title('L2 With optics No Quantization')
subplot(1,3,2)
imshow(X_cs_bef,[])
title('TV L1 With optics No Quantization')
subplot(1,3,3)
imshow(X_cs_dig,[])
title('TV L1 With optics Quantization')
