function rgbImage = nefReadImage(imNames,varargin);

% Returns RGB channel intensities
%
% rgbImage=nefGetChannelsD200(imNames,scaling);
% 
% imNames is either a cell that has full paths for R, G, and B images in 
% sequence or a single path string if all color channels are extracted from
% the same image
%
% scaling is 1 (default) if rgbImage channels are normalized to indicate 
% intensities for an exposure time of 1 sec. 
%
% EXAMPLE:
%
% imNames= ...
% {'/home/parmar/samsung/data/multispectral/20070731/gray/gray_03.nef';...
%  '/home/parmar/samsung/data/multispectral/20070731/gray/gray_04.nef';...
%  '/home/parmar/samsung/data/multispectral/20070731/fabrics/fabrics_01.nef'}
% % 
% rgbImage=nefGetChannelsD200(imNames,1);

if ischar(imNames) % Only 1 NEF file is specified?

    if exist(imNames) % All channels are extracted from same file
        rFile=imNames; gFile=rFile; bFile=rFile;
    else
        error('Input must be a valid Nikon NEF file');        
    end
    
elseif ( iscell(imNames) && length(imNames)==3 ) 
    
    rFile=char(imNames(1)); gFile=char(imNames(2)); bFile=char(imNames(3));
    
    if ~( exist(rFile) && exist(gFile) && exist(bFile) )
        error('Inputs must be valid Nikon NEF files');
    else
        
        % Check if images are consistent
        rData=nefDCrawWrapper(rFile,1);
        gData=nefDCrawWrapper(gFile,1);
        bData=nefDCrawWrapper(bFile,1);

        areSameModel=strcmp(rData.cameraModel,gData.cameraModel) && ...
            strcmp(gData.cameraModel,bData.cameraModel);
        areSameExposureTime=strcmp(rData.shutter,gData.shutter) && ...
            strcmp(gData.shutter,bData.shutter);
        areSameAperture=strcmp(rData.aperture,gData.aperture) && ...
            strcmp(gData.aperture,bData.aperture);
        areSameISO=strcmp(rData.iso,gData.iso) && ...
            strcmp(gData.iso,bData.iso);

        % Will normalize exposure time; other mismatch will give an error
        if ~areSameModel
            error('Channel images are not from same camera');
        end
        if ~areSameAperture
            error('Channel images are acquired with different apertures');
        end  
        if ~areSameISO
            error('Channel images are acquired at different ISOs');
        end
    
    end
else
    error('Invalid input. See help for details.');
end


% Default values for varargin. Change/add stuff later
scaling=1; 
%demosaickingMethod='xxx'

if nargin == 2
    scaling=varargin{1};
    if ~( scaling==0 || scaling==1 )
        error('scaling should be either 1 or 0.');
    end
end


%disp(sprintf('Reading image %d - %s', imageNumber, inFile));
imData = nefDCrawWrapper(rFile);
exposureStr=imData.shutter; % This will be a string that ends with ' sec'
tmp=findstr(exposureStr,'sec');
if ~isempty(tmp)
    exposure=str2num(exposureStr(1:tmp-2));
else
    exposure=1;
end
rImage=double(imData.rawimage)/exposure;

imData = nefDCrawWrapper(gFile);
exposureStr=imData.shutter; % This will be a string that ends with ' sec'
tmp=findstr(exposureStr,'sec');
if ~isempty(tmp)
    exposure=str2num(exposureStr(1:tmp-2));
else
    exposure=1;
end
exposure=str2num(exposureStr(1:tmp-2));
gImage=double(imData.rawimage)/exposure;

imData = nefDCrawWrapper(bFile);
exposureStr=imData.shutter;% This will be a string that ends with ' sec'
tmp=findstr(exposureStr,'sec');
if ~isempty(tmp)
    exposure=str2num(exposureStr(1:tmp-2));
else
    exposure=1;
end
exposure=str2num(exposureStr(1:tmp-2));
bImage=double(imData.rawimage)/exposure;


%% ************** This script is for d200-RGGB **********************   
Ar=rImage; Ar(2:2:end,:)=[]; Ar(:,2:2:end)=[];
Ab=bImage; Ab(1:2:end,:)=[]; Ab(:,1:2:end)=[];
Ag1=gImage; Ag1(2:2:end,:)=[]; Ag1(:,1:2:end)=[];
Ag2=gImage; Ag2(1:2:end,:)=[]; Ag2(:,2:2:end)=[];
Ag=(Ag1+Ag2)/2;

[nRows,nCols]=size(Ar);
rowRange=1:nRows;  colRange=1:nCols;

if nargin==3 % rowRange is not specified
    nRows=length(rowRange);
elseif nargin==4 % rowRange and colRange are specified
    nRows=length(rowRange); nCols=length(colRange);
end

rgbImage=zeros(nRows,nCols,3);
rgbImage(:,:,1)=Ar(rowRange,colRange);
rgbImage(:,:,2)=Ag(rowRange,colRange);
rgbImage(:,:,3)=Ab(rowRange,colRange);
