function rgbImage = ...
        nefGetChannelsD200(imDir,imPrefix,imIndices,rowRange,colRange)
%
% Returns RGB channel intensities for a 1 sec exposure for the Nikon D200.
%
% rgbImage=nefGetChannelsD200(imDir,imPrefix,imIndices,rowRange,colRange);
% 
% imDir, imPrefix and imageIndices are used to parse NEF filenames.
% imIndices is either a vector that has RGB image indices in sequence
%    or a scalar if all color channels are extracted from the same image
% rowRange, colRange are used to define part of image to crop, if they
%    are not specified, the entire image is returned.
%
% EXAMPLE:
%
% imDir= '/home/parmar/samsung/algs/multispectral_images/color_wheel/images_D200';
% imPrefix='image';
% imIndices=27;
% rowRange=927:1026; % Close to the center of the downsampled 
% colRange=605:704; % D200 image (1308x1952)
% rgbImage=nefGetChannelsD200(imDir,imPrefix,imIndices,rowRange,colRange);
%
% TODO:
%   We should probably replace this call with the standard nefReadImage
%   call.

if nargin<3
    error('Please specify image directory, prefix, and index');
end

if length(imIndices)==1, imIndices=repmat(imIndices,1,3); end;

%% Some files may have the prefixes 0 or 00 with the indices. 
rFile=fullfile(imDir,sprintf('%s_%s.nef',imPrefix,num2str(imIndices(1))));
if ~( exist(rFile) == 2 )
    rFile=fullfile(imDir,sprintf('%s_0%s.nef',imPrefix,num2str(imIndices(1))));
    if ~( exist(rFile) == 2 )
        rFile=fullfile(imDir,sprintf('%s_00%s.nef',imPrefix,num2str(imIndices(1))));
    end
end

gFile=fullfile(imDir,sprintf('%s_%s.nef',imPrefix,num2str(imIndices(2))));
if ~( exist(gFile) == 2 )
    gFile=fullfile(imDir,sprintf('%s_0%s.nef',imPrefix,num2str(imIndices(2))));
    if ~( exist(gFile) == 2 )
        gFile=fullfile(imDir,sprintf('%s_00%s.nef',imPrefix,num2str(imIndices(2))));
    end
end

bFile=fullfile(imDir,sprintf('%s_%s.nef',imPrefix,num2str(imIndices(3))));
if ~( exist(bFile) == 2 )
    bFile=fullfile(imDir,sprintf('%s_0%s.nef',imPrefix,num2str(imIndices(3))));
    if ~( exist(bFile) == 2 )
        bFile=fullfile(imDir,sprintf('%s_00%s.nef',imPrefix,num2str(imIndices(3))));
    end
end

%% Get raw images 
%disp(sprintf('Reading image %d - %s', imageNumber, inFile));
imData   = nefDCrawWrapper(rFile);
exposure = findNEFexposure(imData);
rImage   = double(imData.rawimage)/exposure;

imData   = nefDCrawWrapper(gFile);
exposure = findNEFexposure(imData);
gImage   = double(imData.rawimage)/exposure;

imData   = nefDCrawWrapper(bFile);
exposure = findNEFexposure(imData);
bImage   = double(imData.rawimage)/exposure;

%% ************** This script is for d200-RGGB **********************   
Ar=rImage; Ar(2:2:end,:)=[]; Ar(:,2:2:end)=[];
Ab=bImage; Ab(1:2:end,:)=[]; Ab(:,1:2:end)=[];
Ag1=gImage; Ag1(2:2:end,:)=[]; Ag1(:,1:2:end)=[];
Ag2=gImage; Ag2(1:2:end,:)=[]; Ag2(:,2:2:end)=[];
Ag=(Ag1+Ag2)/2;


if nargin==3 % row and colranges are not specified
    [nRows,nCols]=size(Ar);
    rowRange=1:nRows;  colRange=1:nCols;
elseif nargin==4 % colRange is not specified
    nRows=length(rowRange);
elseif nargin==5 % rowRange and colRange are specified
    nRows=length(rowRange); nCols=length(colRange);
end

rgbImage=zeros(nRows,nCols,3);
rgbImage(:,:,1)=Ar(rowRange,colRange);
rgbImage(:,:,2)=Ag(rowRange,colRange);
rgbImage(:,:,3)=Ab(rowRange,colRange);







%%
function exposure=findNEFexposure(imData)

%% imData is the output of nefDCrawWrapper. This function converts the
%% exposure string in metadata to a floating point number

exposureStr=imData.shutter; % This will be a string that ends with ' sec'
tmp=findstr(exposureStr,'sec');
exposure=str2num(exposureStr(1:tmp-2));
return

