% Script
%
% Reads N camera images with 3 (RGB) colors and saves a 3*N band image. The
% exposures are taken with N LEDs. The input are the images acquired at 
% various exposure durations and with various LEDs
%
% The output of this script is a file with a multiband image that contains
% the intensities for each of the 3N channels normalized as if the exposure
% duration was 1 sec.
%
% This script is run manually on a per case basis.  You have to comment and
% uncomment different lines below to create the right file.
%
% Author: Manu Parmar
%
%

%% Specify image specific parameters

dir_image = '/storage-2/samsung/data/multispectral/20070825/veg_calib'
file_prefix='ms_calibration';

nLEDs = 8;  

% Specify the correspondence between the optical filters and which image to
% use for each of the R,G, and B channels.  For example, in the no filter
% case, we use the R,G and B images from image number 99.  In most of these
% cases we use the RGB images from a single snap.  In the last case,
% however, we use the red channel from one image and the blue and green
% from a different image.  This is because of the very large difference in
% exposure times needed for the R and other two channels.
%
% nFilters by 3, stores which image to use for each filter band
% 
 imgForFilter = ...
    [18 18 18;   % UV
     21 21 21;   % Blue
     25 25 25;   % Cyan
     30 30 30;   % Green
     30 30 30;   % Amber
     32 32 32;   % Red
     34 34 34;   % IR_840
     39 39 39];  % IR_940
 
%%
% For each example we need to clip out a region of interest.  These numbers
% are set manually by inspecting the images.  The region is selected to
% include the plate of fruit, or the object of interest.  

% You can select the crop this way
cropImage = nefGetChannelsD200(dir_image,file_prefix,imgForFilter(3,:));
[tmp,rect] = imcrop(cropImage/max(cropImage(:)));
rect = round(rect);
colRange = rect(1):(rect(1)+rect(3));
rowRange = rect(2):(rect(2)+rect(4));
% 
% rowRange = 315:900;  
% colRange = 300:1400; 

nRows=length(rowRange);
nCols=length(colRange);

%% Process the images

% Allocate the output array
filterImages=zeros(nRows,nCols,3*nFilters);

% For each optical filter, read in the image data that we will save.
for kk=1:nFilters
    disp(sprintf('Reading data for Filter %d',kk));

    rgbImage=...
      nefGetChannelsD200(dir_image,file_prefix,imgForFilter(kk,:),rowRange,colRange);
    
    bandStart=(kk-1)*3+1;
    
    filterImages(:,:,bandStart:bandStart+2)=rgbImage;
end

% To preview and check the data
for kk=1:3*nFilters
    imagesc(filterImages(:,:,kk)); colormap(gray(256)); axis image;
    title(sprintf('%d',kk))
    pause(1);
end

%
% save filterImagesBrian filterImages
outFile = sprintf('/storage-3/filterImages/filterImages_%s.mat',file_prefix);
save(outFile, 'filterImages')

% save filterImagesSerge filterImages
% save filterImagesJoyce filterImages
% save filterImagesMichal filterImages
% save filterImagesJon filterImages
%% Other images, other days
%
% The user must select and edit these

