function [out, norm_c] = bilateral_filter3(in, sig_d, sig_r, opt1, opt2, white_xyz, XYZ, phosphor)

%BILATERAL_FILTER3 implements the bilateral filter algorithm for various 
%color spaces including the RGB, YCbCr, HSV and CIELAB spaces. 
%
%Inputs: in - input image (m-by-n-by-3 rgb image)
%        sig_d - geometric radius for domain filtering
%        sig_r - photometric radius  for range filtering
%        opt1 - option that specifies what space to carry the smoothing
%               over. opt1 could be 'rgb', 'ycb', 'hsv', or 'lab'
%        opt2 - option that specifies what space to do the range filtering
%               in. opt2 could be 'rgb', 'ycb', 'hsv', or 'lab'
%        white_xyz - the 1-by-3 vector that specifies the white point for
%                    conversion to the CIELAB space. needed when opt1 or 
%                    opt2 is set to 'lab'
%        XYZ - the XYZ matrix used in the conversion from 'rgb' to 'xyz'.
%              needed when opt1 or opt2 is set to 'lab'
%        phosphor - the matrix that specifies the phosphor intensity of
%                   the display monitor. needed when opt1 or opt2 is set 
%                   to 'lab'
%
%Outputs: out - output image (m-by-n-by-3 rgb image)
%         norm_c - normalization factor used in filtering
%
%%%%%%%%%%%                                                   %%%%%%%%%%%%%
%%%%%%%%%%% Copyright 2006 Timothy Atobatele & Ekine Akuiyibo.%%%%%%%%%%%%% 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%convert image to double
in = im2double(in);

if opt1 == 'rgb'
    in1 = in; op1 = 1;
elseif opt1 == 'ycb'
    in1 = rgb2ycbcr(in); op1 = 2;
elseif opt1 == 'hsv'
    in1 = rgb2hsv(in); op1 = 3;
elseif opt1 == 'lab'
    rgb2xyz = XYZ'*phosphors; op1 = 4;
    xyz2rgb = inv(rgb2xyz);
    in_rgb = reshape(in, size(in, 1)*size(in, 2), 3);
    in_xyz = (rgb2xyz*in_rgb')';   
    in_lab = vcXYZ2lab(in_xyz, white_xyz);
    in1 = reshape(in_lab, size(in, 1),size(in, 2), 3);
else
    error('Invalid entry for opt1. See documentation. \n');
end

if opt2 == 'rgb'
    in2 = in; op2 = 1;
elseif opt2 == 'ycb'
    in2 = rgb2ycbcr(in); op2 = 2;
elseif opt2 == 'hsv'
    in2 = rgb2hsv(in); op2 = 3;
elseif opt2 == 'lab'
    rgb2xyz = XYZ'*phosphors; op2 = 4;
    xyz2rgb = inv(rgb2xyz);
    in_rgb = reshape(in, size(in, 1)*size(in, 2), 3);
    in_xyz = (rgb2xyz*in_rgb')';   
    in_lab = vcXYZ2lab(in_xyz, white_xyz);
    in2 = reshape(in_lab, size(in, 1),size(in, 2), 3);
else
    error('Invalid entry for opt2. See documentation. \n');
end
 

d = 1/(2*sig_d^2);
r = 1/(2*sig_r^2);


row_in = size(in1,1);
col_in = size(in1,2);


%window size for adaptive smoothing is (2w+1)-by-(2w+1)
w = max(1, round(3*sig_d)); 

for i = 1:row_in
    for j = 1:col_in
         
        rlb = max(1, -w+i);
        rub = rlb + 2*w;
        
        if rub > row_in
            rub = row_in;
            rlb = rub - 2*w;
        end
        
        clb = max(1, -w+j);
        cub = clb + 2*w;
        if cub > col_in
            cub = col_in;
            clb = cub - 2*w;
        end
        
        row_vec = rlb:rub;
        col_vec = clb:cub;
        
        lr = length(row_vec);
        lc = length(col_vec);       
        
        %Form a row_vec-by-col_vec window of pixels around the pixel (i,j) 
        %on which adaptive smoothing would be performed
        in_win1 = in1(row_vec, col_vec,:); 
        in_win2 = in2(row_vec, col_vec,:); 
        
        %A matrix of Euclidean distances squared between the neighboring pixels 
        %and the pixel (i,j) for domain filtering  
        dist1 = (row_vec'*ones(1,lr) - ones(lr, lr)*i).^2 +...
            (ones(lc,1)*col_vec - ones(lc, lc)*j).^2 ;
        
        
        %Domain Filter Kernel: exp(-d*dist1)
        %Range Filer Kernel: exp(-r*dist2)
        if op2 == 3 || op2 == 4
            in2_c = [in2(i,j,1) in2(i,j,2) in2(i,j,3)];
            in_win2_r = reshape(in_win2, lr*lc, 3);
            dist2_r = sum( (in_win2_r - ones(lr*lc, 1)*in2_c).^2, 2);
            dist2 = reshape(dist2_r, lr, lc);
            for k=1:3
                p_out = in_win1(:,:,k).*exp(-d*dist1).*exp(-r*dist2); 
                p_norm_c = exp(-d*dist1).*exp(-r*dist2);
                out(i,j,k) = sum(p_out(:)); %convolution sum
                norm_c(i,j,k) = sum(p_norm_c(:)); %normalization factor
                out(i,j,k) = out(i,j,k)/norm_c(i,j,k); %the smooth output pixel (i,j)
            end
        elseif op2 == 1 || op2 == 2
            for k=1:3
                dist2 = (in_win2(:,:,k) - ones(lr,lc)*in2(i,j,k)).^2;
                p_out = in_win1(:,:,k).*exp(-d*dist1).*exp(-r*dist2); 
                p_norm_c = exp(-d*dist1).*exp(-r*dist2);
                out(i,j,k) = sum(p_out(:)); %convolution sum
                norm_c(i,j,k) = sum(p_norm_c(:)); %normalization factor
                out(i,j,k) = out(i,j,k)/norm_c(i,j,k); %the smooth output pixel (i,j)
            end
        else
            error('error in calculating dist2 \n');
        end           
        
    end
    clc
    fprintf('%f percent completed\n', (i/row_in)*100);
end


if opt1 == 'rgb'
            out = out;
elseif opt1 == 'ycb'
            out = ycbcr2rgb(out);
elseif opt1 == 'hsv'
            out = hsv2rgb(out);
elseif opt1 == 'lab' 
    out1 = reshape(out, size(out, 1)*size(out,2), 3);
    out2 = lab2xyz(out1, white_xyz);
    out3 = (xyz2rgb*out2')';
    out = reshape(out3, size(out, 1), size(out,2), 3);
else
    error('error in calculating out \n');
end 
