function [pyr,masks,confidences,stds,means] = preprocess(filename, ...
							 distances, K, LAB_DIFF, PERCENTAGE, PERCEPTUAL)

% PREPROCESS - applies the four steps of pre-processing before relaxation starts
%
%  [MASKS,CONFIDENCES,STDS,MEANS] = PREPROCESS(FILENAME,K,DISTANCES,LAB_DIFF,PERCENTAGE) 
%
%  Input:
%       FILENAME - path and name for the input image
%       DISTANCES - vector of distances (in inches) used for performing the perceptual filtering
%       K - used for K-means clustering. This is the initial number of clusters to start with.
%       LAB_DIFF - a constant used in clusterMerge.m to specify the threshold for merging neighboring clusters
%       PERCENTAGE - constant used in clusterMerge.m to specify the threshold for removing insignificant clusters
%
%       PERCEPTUAL - A flag, which if set to 1, uses the regular
%       SCIELAB filters in the opponent color space.  If set to 0,
%       uses an average filter in all color planes.
%  Output: 
% 	PYR - MxNxL matrix containing the perceptual pyramid
%       MASKS - MxNxC set of binary masks to specify pixel membership for each cluster
%       CONFIDENCES - MxNxC matrix, where the (i,j,k) entry is the probability that the (i,j pixel belongs to 
%                     the kth cluster
%       STDS - Cx1 vector containing the standard deviation of distances from the mean for each cluster
%       MEANS - Cx3 vector containing the mean in LAB space for each cluster
%
% Jeff Walters & Angi Chau
% Feb 2003

orig = imread(filename,'JPG');
im = double(orig)./255;
[h,w,c]= size(im);

disp(sprintf('preprocess: perceptual = %d',PERCEPTUAL));

% STEP 1: perceptual blurring 
disp('STEP 1 - APPLYING PERCEPTUAL FILTERS');
M=createPyramid(im, distances,PERCEPTUAL);
showPyramid(M, distances);


% STEP 2: k-means cluster on coarsest level
disp('STEP 2 - KMEANS CLUSTERING ON COARSEST LEVEL');
lowimg = M(:,:,:,end);

% display this image for reference
RGB_WHITE = [1 1 1]';
figure;
whiteXYZ = changeColorSpace(RGB_WHITE, cmatrix('rgb2xyz'));
whiteXYZ = whiteXYZ./whiteXYZ(2)*100;
load displayGamma;
thisXYZ = lab2xyz(lowimg, whiteXYZ);
imgLinearRGB = changeColorSpace(thisXYZ, cmatrix('xyz2rgb'));
imgRGB = dac2rgb(imgLinearRGB,invGamma);
imagesc(imgRGB./255);
axis image;
axis off;
title('coarsest level');

% do kmeans
[Kmasks,Kmeans,Kgroups,Kmap] = kmeansClusters(lowimg, K); 


% STEP 3: merging clusters
disp('STEP 3 - MERGING CLUSTERS');
[Mmasks Mgroups Mmap Mmeans] = clusterMerge(lowimg,Kgroups,Kmap,Kmasks,Kmeans(:,3:5),LAB_DIFF,PERCENTAGE);
figure
colors=colormap(bone(size(Mmasks,3)+1));
cmap = zeros(max(Mgroups(:))+1,3);
count=size(colors,1);
cmapind=unique(Mgroups(:))
for ind=1:count
    cmap(cmapind(ind)+1,:) = colors(ind,:);
end
colormap(cmap);
imagesc(Mgroups);
axis image;
axis off;
title('merged clusters') 


% STEP 4: core clusters
disp('STEP 4 - FORMING CORE CLUSTERS');
[Cstds,Cmeans,Cconfs,Cmasks]=formCoreClusters(lowimg,Mmasks);
numClusters=size(Cmasks,3);
% figure;
% showme = zeros(h,w);
% for ind=1:numClusters
%     indices=find(Cmasks(:,:,ind)~=0);
%     showme(indices) = ind;
% end
% imagesc(showme)
% colormap(bone(numClusters+1))
% colorbar;
% axis image;
% axis off;
showClusters(Cmasks);
title('core clusters')


% STEP 5: output
masks = Cmasks;
confidences = Cconfs;
stds = Cstds;
means = Cmeans;
pyr = M;


