function [stds,mus,probs,newClusters] = formCoreClusters(im, initClusters)
 
% FORMCORECLUSTERS - using the statistics of each cluster, calculate
% probabilities of each pixel being in that cluster
%
% Input:
%       IM - MxNx3 image, where each pixel is a point in a
%       perceptually uniform color space.
%
%       INITCLUSTERS - MxNxC matrix that contains binary masks
%       representing a pixels membership in a certain cluster.
%
% Output:
%       STDS - Cx1 vector containing the stds of the final clusters
%       MUS - Cx3 matrix containing the means in LAB space of the final clusters
%       PROBS - MxNxC matrix containing the confidence levels of each pixel
%               being associated with cluster c
%       newClusters - MxNxC matrix containing binary masks for the new clusters
%               using a defined threshold level for confidences.
%
% Jeff Walters & Angi Chau
% Feb 2003

CONFIDENCE_THRESHOLD = 0.8;

[h,w,c] = size(im);

% for each cluster, we want to find the std and mean
numClusters = size(initClusters, 3);
stdClusters = zeros(numClusters, 1);
meanClusters = zeros(numClusters, 3);
conf = zeros(h,w,numClusters);

for ind=1:numClusters
    disp(sprintf('looking at cluster %d',ind));
    maskedim = zeros(h,w,3);
    
    % using the mask, get the pixels associated with this cluster
	maskedim(:,:,1) = initClusters(:,:,ind).*im(:,:,1);	
	maskedim(:,:,2) = initClusters(:,:,ind).*im(:,:,2);
	maskedim(:,:,3) = initClusters(:,:,ind).*im(:,:,3);

	% calculate the cluster's mean and std using these pixels.
	[thisStd, thisMean]=statsCluster(maskedim, initClusters(:,:,ind));
	stdClusters(ind) = thisStd;
	meanClusters(ind,:) = thisMean;
    disp(sprintf('std = %f, mean=[%f %f %f]',thisStd, thisMean(1), thisMean(2), thisMean(3)));

	% this is just the numerator in the phat_c^i right now,
	% still need to divide by the sum across all clusters
	conf(:,:,ind) = thisStd^2 ./ (sqdist(im,thisMean) + thisStd^2);

end

% sum up all confidences across clusters
sumConfs = sum(conf,3);
disp('normalizing all confidences');

% now we need to normalize. for each pixel, divide the conf by
% the sum of conf across all clusters for that pixel
for ind=1:numClusters,
	conf(:,:,ind) = conf(:,:,ind) ./ sumConfs;
end 

% form a set of new masks and confidences by taking out any empty regions
newClusters = zeros(h,w,0);
probs = zeros(h,w,0);
stds = zeros(0,1);
mus = zeros(0,size(meanClusters,2));

for j=1:numClusters,
    % go thru each plane of confidences and pick out the ones with conf > THRESHOLD
    mask = (conf(:,:,j) >= CONFIDENCE_THRESHOLD);
    if (sum(mask(:)) ~= 0)
        disp(sprintf('cluster %d is not empty --> adding to output',j));
        newClusters(:,:,end+1) = mask;
        probs(:,:,end+1) = conf(:,:,j);
        stds(end+1,:) = stdClusters(j);
        mus(end+1,:) = meanClusters(j,:);
    end
end


